<?php

namespace App\Http\Controllers;

use App\Models\BranchModel;
use App\Models\InvoiceItemModel;
use App\Models\InvoiceModel;
use App\Models\LoyaltyTransactionModel;
use App\Models\PaymentsModel;
use App\Models\ProductModel;
use App\Models\User;
use Carbon\Carbon;
use Faker\Provider\ar_EG\Payment;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        // Basic counts
        $staff_count = User::where('branch_id', session('branch_id'))->where("user_flg", "S")->count();
        $product_count = ProductModel::where('branch_id', session('branch_id'))->count();
        // $products_sold = InvoiceItemModel::sum('quantity');
        $invoice_count = InvoiceModel::where('branch_id', session('branch_id'))->whereDate('invoice_date', Carbon::today())->count();
        $total_sales = InvoiceModel::where('branch_id', session('branch_id'))->sum('final_amount');

        $branch = BranchModel::where('id', session('branch_id'))->first();

        $invoices = InvoiceModel::with([
            'payments' => function ($query) {
                $query->whereDate('created_at', Carbon::today());
            }
        ])
            ->where('branch_id', session('branch_id'))
            ->get();

        $cashTransaction = $invoices->sum(function ($invoice) {
            return $invoice->payments->where('payment_mode', 3)->sum('amount');
        });

        $bankTransaction = $invoices->sum(function ($invoice) {
            return $invoice->payments->whereIn('payment_mode', [1, 2])->sum('amount');
        });

        // Monthly data
        $product_monthly_sales = InvoiceItemModel::selectRaw('MONTH(invoice_items.created_at) as month, sum(invoice_items.quantity) as quantity_sold')
            ->join('products', 'invoice_items.product_id', '=', 'products.id')
            ->groupBy('month')
            ->pluck('quantity_sold', 'month')
            ->toArray();

        $sales_monthly = InvoiceModel::selectRaw('MONTH(created_at) as month, sum(final_amount) as total_sales')
            ->groupBy('month')
            ->pluck('total_sales', 'month')
            ->toArray();

        // Today's invoices with items and taxes
        $results = InvoiceModel::with(['invoiceItems.products', 'invoiceItems.invoiceItemTaxes'])
            ->where('oprntl_flag', 'A')
            ->where('branch_id', session('branch_id'))
            ->get()
            ->filter(function ($invoice) {
                return Carbon::parse($invoice->invoice_date)->isToday();
            });

        // Initialize totals
        $profitDetails = [];
        $todayTotalSellingPrice = 0;
        $todayTotalMarginPrice = 0;
        $todayTotalProfit = 0;
        $todayTotalTaxAmount = 0;
        $redeem = 0;

        foreach ($results as $invoice) {
            $invoiceTotalSellingPrice = 0;
            $invoiceTotalMarginPrice = 0;
            $invoiceTotalTaxAmount = 0;
            $productProfitDetails = [];

            $points = LoyaltyTransactionModel::where('reference_invoice', $invoice->invoice_id)->where('type', 'redeem')->first();

            $discount = 0;
            if (isset($invoice->discount)) {
                $discount = $invoice->discount;
            }

            foreach ($invoice->invoiceItems as $invoiceItem) {
                $product = $invoiceItem->products;
                $sellingPrice = $product->unit_price;
                $marginPrice = $product->margin_price;
                $quantity = $invoiceItem->quantity;

                $taxAmount = 0;
                foreach ($invoiceItem->invoiceItemTaxes as $tax) {
                    $taxAmount += ($tax->tax / 100) * $sellingPrice * $quantity;
                }

                $itemTotalSellingPrice = ($sellingPrice * $quantity) + $taxAmount;
                $itemTotalMarginPrice = $marginPrice * $quantity;

                $invoiceTotalSellingPrice += $itemTotalSellingPrice;
                $invoiceTotalMarginPrice += $itemTotalMarginPrice;
                $invoiceTotalTaxAmount += $taxAmount;

                $productProfitDetails[] = [
                    'product_name' => $product->name,
                    'selling_price' => $sellingPrice,
                    'margin_price' => $marginPrice,
                    'quantity' => $quantity,
                    'tax_amount' => $taxAmount,
                ];
            }

            // Apply discount
            if ($invoice->discount_type == 1) {
                // Flat discount
                $invoiceTotalSellingPrice -= $discount;
            } else {
                // Percentage discount
                $discountAmount = $invoiceTotalSellingPrice * ($invoice->discount / 100);
                $invoiceTotalSellingPrice -= $discountAmount;
            }

            $profitDetails[] = [
                'invoice_id' => $invoice->invoice_id,
                'invoice_date' => $invoice->invoice_date,
                'total_selling_price' => $invoiceTotalSellingPrice,
                'total_margin_price' => $invoiceTotalMarginPrice,
                'tax_amount' => $invoiceTotalTaxAmount,
                'product_profit_details' => $productProfitDetails,
            ];

            // Add to today's totals
            //$redeem += max(floor($points->points * $branch->point_redemption), 2);
            if ($points) {
                $redeem += max(floor($points->points * $branch->point_redemption), 2);
            }
            $todayTotalSellingPrice += max(floor($invoiceTotalSellingPrice), 2);
            $todayTotalMarginPrice += max(floor($invoiceTotalMarginPrice), 2);
            $todayTotalTaxAmount += max(floor($invoiceTotalTaxAmount), 2);
            $todayTotalProfit += max(floor($invoiceTotalSellingPrice - $invoiceTotalMarginPrice - $invoiceTotalTaxAmount), 2);
        }


        // Return or pass these values to a view as needed
        return view('dashboard.dashboard', compact(
            'staff_count',
            'product_count',
            // 'products_sold',
            'invoice_count',
            'total_sales',
            'product_monthly_sales',
            'sales_monthly',
            'profitDetails',
            'todayTotalSellingPrice',
            'todayTotalMarginPrice',
            'todayTotalTaxAmount',
            'todayTotalProfit',
            'cashTransaction',
            'bankTransaction',
            'redeem',
            'branch',
        ));
    }

}