<?php

namespace App\Http\Controllers;

use App\Models\AssembledItemModel;
use App\Models\BOMItemsModel;
use App\Models\BOMModel;
use App\Models\CategoryModel;
use App\Models\CostTypeModel;
use App\Models\InvoiceItemModel;
use App\Models\OrdersItemsModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductHistoryModel;
use App\Models\ProductModel;
use App\Models\ProductTypeModel;
use App\Models\QuotesItemsModel;
use App\Models\SupplierModel;
use App\Models\UomModel;
use Auth;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ManufactureController extends Controller
{
    public function index()
    {
        $bom = BOMModel::with('bomItems')->whereIn('oprntl_flag', ['A', 'C'])->where('branch_id', session('branch_id'))->get();
        $assembledItem = AssembledItemModel::with('bom')->where('oprntl_flag', 'A')->where('branch_id', session('branch_id'))->get();
        return view('manufacture.index', compact('bom', 'assembledItem'));
    }

    public function bomAdd()
    {
        $product_type_id = ProductTypeModel::whereIn('name', ['Raw Material', 'Both'])
            ->pluck('id')
            ->toArray();

        $product = ProductModel::with('category.unitOfMass')
            ->whereIn('product_type', $product_type_id)
            ->where('branch_id', session('branch_id'))
            ->get();
        $unit = UomModel::get();
        $costType = CostTypeModel::where('oprntl_flag', 'A')->get();
        return view('manufacture.bom.add', compact('product', 'unit', 'costType'));
    }
    public function bomEdit($id)
    {
        $product_type_id = ProductTypeModel::where('name', 'Raw Material')->value('id');
        $bom = BOMModel::with('bomItems.product', 'bomItems.unit', 'bomItems.costType')->where('id', $id)->where('branch_id', session('branch_id'))->first();
        $product = ProductModel::with('category.unitOfMass')->where('product_type', $product_type_id)->where('branch_id', session('branch_id'))->get();
        $unit = UomModel::get();
        $costType = CostTypeModel::where('oprntl_flag', 'A')->get();

        return view('manufacture.bom.edit', compact('bom', 'product', 'unit', 'costType'));
    }

    public function bomStore(Request $request)
    {
        // dd($request->all());
        $validated = $request->validate([
            'assembled_item' => 'required|string',
            'bom_items' => 'required|array|min:1',
            'additional_costs' => 'nullable|array',
        ]);

        DB::beginTransaction();
        try {
            $bom = BOMModel::create([
                'branch_id' => session('branch_id'),
                'assemble_item' => $validated['assembled_item'],
                'remarks' => $validated['remarks'] ?? null,
                'created_by' => Auth::user()->id,
            ]);


            foreach ($validated['bom_items'] as $item) {
                BOMItemsModel::create([
                    'bom_id' => $bom->id,
                    'product_id' => $item['product_id'],
                    'uom_id' => $item['uom_id'],
                    'qty' => $item['qty'],
                    'unit_cost' => $item['unit_cost'],
                    'amount' => $item['amount'],
                    'type' => 'product'
                ]);

            }

            if (!empty($validated['additional_costs'])) {
                foreach ($validated['additional_costs'] as $cost) {
                    BOMItemsModel::create([
                        'bom_id' => $bom->id,
                        'qty' => 1,
                        'uom_id' => $cost['uom_id'],
                        'unit_cost' => $cost['amount'],
                        'amount' => $cost['amount'],
                        'type' => $cost['cost_type_id'],
                        'description' => $cost['description'] ?? $cost['cost_type_text'],
                    ]);

                }
            }

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => 'Bill of Material Saved Successfully.',
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error while storing the bill of material: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong!',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function bomUpdate(Request $request, $id)
    {
        $validated = $request->validate([
            'assembled_item' => 'required|string',
            'bom_items' => 'required|array|min:1',
            'additional_costs' => 'nullable|array',
        ]);

        DB::beginTransaction();
        try {
            $bom = BOMModel::findOrFail($id);

            $bom->update([
                'assemble_item' => $request->assembled_item,
                'remarks' => $request->remarks,
            ]);

            BOMItemsModel::where('bom_id', $bom->id)->delete();

            foreach ($validated['bom_items'] as $item) {
                BOMItemsModel::create([
                    'bom_id' => $bom->id,
                    'product_id' => $item['product_id'],
                    'uom_id' => $item['uom_id'],
                    'qty' => $item['qty'],
                    'unit_cost' => $item['unit_cost'],
                    'amount' => $item['amount'],
                    'type' => 'product'
                ]);

            }

            if (!empty($validated['additional_costs'])) {
                foreach ($validated['additional_costs'] as $cost) {
                    BOMItemsModel::create([
                        'bom_id' => $bom->id,
                        'qty' => 1,
                        'uom_id' => $cost['uom_id'],
                        'unit_cost' => $cost['amount'],
                        'amount' => $cost['amount'],
                        'type' => $cost['cost_type_id'],
                        'description' => $cost['description'] ?? $cost['cost_type_text']
                    ]);

                }
            }

            DB::commit();
            return response()->json([
                'status' => 'success',
                'message' => 'Bill of Material Updated Successfully.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('while updating the bill of material :' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Something went wrong while updating!',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function bomDelete($id)
    {
        $bom = BOMModel::with('bomItems')->findOrFail($id);

        $isUsedInAssembly = AssembledItemModel::where('bom_id', $bom->id)->exists();

        if ($isUsedInAssembly) {
            return redirect()->route('manufacture-index')->with('error', 'Cannot delete BOM. It is already used in an assembled item.');
        }

        $bom->bomItems()->delete();
        $bom->delete();

        return redirect()->route('manufacture-index')->with('success', 'Bill Of Material Deleted Successfully');
    }


    public function assembleAdd()
    {
        $bom = BOMModel::with('bomItems.costType')->where('oprntl_flag', 'A')->where('branch_id', session('branch_id'))->get();
        $category = CategoryModel::where('branch_id', session('branch_id'))->get();
        $suppliers = SupplierModel::where('branch_id', session('branch_id'))->where('active_status', 1)->get();

        return view('manufacture.assembled.add', compact('bom', 'category', 'suppliers'));
    }
    public function assembleEdit($id)
    {
        $assemble = AssembledItemModel::with('product')->findOrFail($id);
        $bom = BOMModel::with('bomItems.costType')->whereIn('oprntl_flag', ['A', 'C'])->where('branch_id', session('branch_id'))->get();
        $category = CategoryModel::where('branch_id', session('branch_id'))->get();
        $suppliers = SupplierModel::where('branch_id', session('branch_id'))->where('active_status', 1)->get();

        return view('manufacture.assembled.edit', compact('bom', 'category', 'bom', 'suppliers', 'assemble'));
    }

    public function assembleStore(Request $request)
    {
        $request->validate([
            'assembled_date' => 'required|date',
            'bom_id' => 'required|exists:bom,id',
            'assembled_qty' => 'required',
            'code' => 'required|unique:products,code,' . $request->previous_product . ',id,branch_id,' . session('branch_id'),
            'category_id' => 'required|exists:categories,id',
            'selling_price' => 'required'
        ]);

        DB::beginTransaction();
        try {
            foreach ($request->bom_items as $bomItem) {
                $product = ProductModel::findOrFail($bomItem['product_id']);
                if ($product->qty < $bomItem['qty']) {
                    \Log::error("Insufficient stock for {$product->name}. Available: {$product->qty}, Required: {$bomItem['qty']}");
                    return redirect()->back()->with('error', "Insufficient stock for {$product->name}. Available: {$product->qty}, Required: {$bomItem['qty']}")->withInput();
                }
            }

            $bom = BOMModel::where('id', $request->bom_id)->where('oprntl_flag', 'A')->first();
            $item = $bom->assemble_item;
            $assembledItem = AssembledItemModel::create([
                'branch_id' => session('branch_id'),
                'assemble_date' => $request->assembled_date,
                'bom_id' => $request->bom_id,
                'assembled_item' => $item,
                'assembled_qty' => $request->assembled_qty,
                'remarks' => $request->remarks,
                'per_unit_cost' => $request->per_unit_cost,
                'total_amount' => $request->grand_total,
                'created_by' => Auth::user()->id,
            ]);

            $details_code = $this->generateProductDetailCode();
            if (!empty($request->previous_product)) {
                $product = ProductModel::where('id', $request->previous_product)->first();
                $newQty = $product->qty + $request->assembled_qty;
                $newTotalQty = $product->total_qty + $request->assembled_qty;
                $product->update([
                    'unit_price' => $request->selling_price,
                    'qty' => $newQty,
                    'total_qty' => $newTotalQty,
                    'description' => $request->remarks ?? null,
                ]);
            } else {
                $product = ProductModel::create([
                    'name' => $item,
                    'code' => $request->code,
                    'hsn_code' => $request->code,
                    'category_id' => $request->category_id,
                    'unit_price' => $request->selling_price,
                    'description' => $request->remarks ?? null,
                    'branch_id' => session('branch_id'),
                    'margin_price' => $request->per_unit_cost,
                    'qty' => $request->assembled_qty,
                    'total_qty' => $request->assembled_qty,
                    'product_type' => 2,
                    'assembled_item_id' => $assembledItem->id,
                ]);
            }
            $product_id = $product->id;

            $assembledItem->update(['product_id' => $product_id]);

            ProductDetailsModel::create([
                'product_id' => $product_id,
                'product_details_code' => $details_code,
                'name' => $item,
                'code' => $request->code,
                'category_id' => $request->category_id,
                'unit_price' => $request->selling_price,
                'margin_price' => $request->per_unit_cost,
                'description' => $request->remarks ?? null,
                'branch_id' => session('branch_id'),
                'qty' => $request->assembled_qty,
                'total_amount' => $request->grand_total,
                'status' => 2,
                'assembled_item_id' => $assembledItem->id,
            ]);

            foreach ($request->bom_items as $bomItem) {
                $product = ProductModel::find($bomItem['product_id']);
                if ($product) {
                    $newQty = $product->qty - $bomItem['qty'];

                    if ($newQty < 0) {
                        DB::rollBack();
                        return redirect()->back()->with('error', "Not enough stock for raw material: {$product->name}.");
                    }

                    $product->update([
                        'qty' => $newQty,
                    ]);
                }
            }

            ProductHistoryModel::create([
                'product_id' => $product_id,
                'branch_id' => session('branch_id'),
                'name' => $item,
                'code' => $request->code,
                'category_id' => $request->category_id,
                'unit_price' => $request->selling_price,
                'qty' => $request->assembled_qty,
                'description' => $data['description'] ?? null,
                'product_image' => $request->remarks ?? null,
                'action' => 'A',
                'created_by' => auth()->user()->id,
            ]);

            DB::commit();

            return redirect()->route('manufacture-index')->with('success', 'Assembled Item Saved Successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('While Saving Assembled Items: ' . $e->getMessage());
            return redirect()->route('manufacture-index')->with('error', 'Failed to Save Assembled Item.');
        }
    }

    public function assembleUpdate(Request $request, $id)
    {
        $request->validate([
            // 'assembled_qty' => 'required|numeric|min:1',
            // 'category_id' => 'required|exists:categories,id',
            'selling_price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            $assembledItem = AssembledItemModel::findOrFail($id);

            $oldAssembledQty = $assembledItem->getOriginal('assembled_qty');
            $newAssembledQty = $request->assembled_qty;

            $bom = BOMModel::with('bomItems')->findOrFail($assembledItem->bom_id);

            foreach ($bom->bomItems as $item) {
                $product = ProductModel::find($item->product_id);
                if ($product) {
                    $restoreQty = $item->qty * $oldAssembledQty;
                    $product->update([
                        'qty' => $product->qty + $restoreQty
                    ]);
                }
            }

            foreach ($bom->bomItems as $item) {
                $product = ProductModel::find($item->product_id);
                if ($product) {
                    $deductQty = $item->qty * $newAssembledQty;

                    if ($product->qty < $deductQty) {
                        DB::rollBack();
                        return redirect()->back()->with('error', "Insufficient stock for raw material: {$product->name}");
                    }

                    $product->update([
                        'qty' => $product->qty - $deductQty
                    ]);
                }
            }

            $assembledItem->update([
                // 'assembled_qty' => $newAssembledQty,
                'remarks' => $request->remarks,
                // 'total_amount' => $request->grand_total,
            ]);

            $product = ProductModel::findOrFail($assembledItem->product_id);
            $qtyDifference = $newAssembledQty - $oldAssembledQty;

            if (($product->qty + $qtyDifference) < 0) {
                DB::rollBack();
                return redirect()->back()->with(
                    'error',
                    "Cannot reduce assembled quantity below available stock. " .
                    "Current stock: {$product->qty}, Attempting to reduce by: " . abs($qtyDifference)
                );
            }


            $product->update([
                // 'category_id' => $request->category_id,
                'unit_price' => $request->selling_price,
                'description' => $request->remarks ?? null,
                // 'qty' => $product->qty + $qtyDifference,
                // 'total_qty' => $product->total_qty + $qtyDifference,
            ]);

            $productDetails = ProductDetailsModel::where('product_id', $product->id)
                ->where('assembled_item_id', $assembledItem->id)
                ->first();

            if ($productDetails) {
                $productDetails->update([
                    // 'category_id' => $request->category_id,
                    'unit_price' => $request->selling_price,
                    'description' => $request->remarks ?? null,
                    // 'qty' => $productDetails->qty + $qtyDifference,
                    // 'total_amount' => $request->grand_total,
                ]);
            }

            ProductHistoryModel::create([
                'product_id' => $product->id,
                'branch_id' => session('branch_id'),
                'name' => $product->name,
                'code' => $product->code,
                'category_id' => $product->category_id,
                'unit_price' => $product->unit_price,
                'qty' => $product->qty,
                'description' => $product->description ?? null,
                'product_image' => $product->products_image ?? null,
                'action' => 'E',
                'created_by' => auth()->user()->id,
            ]);

            DB::commit();
            return redirect()->route('manufacture-index')->with('success', 'Assembled Item Updated Successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('While Updating Assembled Item: ' . $e->getMessage());
            return redirect()->route('manufacture-index')->with('error', 'Failed to Update Assembled Item.');
        }
    }

    public function assembleDelete($id)
    {
        DB::beginTransaction();
        try {
            $assemble = AssembledItemModel::with('bom.bomItems')->findOrFail($id);

            $assembledQty = $assemble->assembled_qty;

            foreach ($assemble->bom->bomItems as $item) {
                $product = ProductModel::find($item->product_id);
                if ($product) {
                    $restoreQty = $item->qty * $assembledQty;
                    $product->update([
                        'qty' => $product->qty + $restoreQty
                    ]);
                }
            }

            $product = ProductModel::with('details')->find($assemble->product_id);
            if ($product) {
                $usedInInvoices = InvoiceItemModel::where('product_id', $product->id)->exists();

                $usedInQuotes = QuotesItemsModel::where('product_id', $product->id)->exists();

                $usedInOrders = OrdersItemsModel::where('product_id', $product->id)->exists();

                if ($usedInInvoices || $usedInQuotes || $usedInOrders) {
                    return redirect()->back()->with('error', 'Cannot delete this assembled item. Product has already been used in transactions.');
                }

                $product->details()->delete();

                $product->delete();
            }

            // $assemble->bom->update([
            //     'oprntl_flag' => 'A',
            // ]);

            $assemble->delete();

            DB::commit();
            return redirect()->route('manufacture-index')->with('success', 'Assembled Item Deleted and stock restored successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('While deleting assembled item: ' . $e->getMessage());
            return redirect()->route('manufacture-index')->with('error', 'Failed to delete assembled item.');
        }
    }


    private function generateProductDetailCode(): string
    {
        $lastDetail = ProductDetailsModel::orderBy('product_details_sid', 'desc')->first();

        if ($lastDetail && $lastDetail->product_details_code) {
            $lastCode = $lastDetail->product_details_code;
            $parts = explode('-', $lastCode);
            $prefix = $parts[0];
            $number = isset($parts[1]) ? (int) $parts[1] : 0;
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        return 'PROD-001';
    }

    public function getProducts(Request $request)
    {
        $bomId = $request->bomId;

        $assembledItemId = AssembledItemModel::where('bom_id', $bomId)->value('id');
        $product = null;
        if ($assembledItemId) {
            $product = ProductModel::where('assembled_item_id', $assembledItemId)
                ->first();
        }

        $bomItems = BOMItemsModel::with(['product', 'unit', 'costType'])
            ->where('bom_id', $bomId)
            ->get();

        return response()->json([
            'bom_items' => $bomItems,
            'product' => $product ?? null,
            'unit_price' => $bomItems->sum('amount'),
        ]);
    }


    public function saveCostType(Request $request)
    {
        $costType = CostTypeModel::create([
            'name' => $request->costType,
        ]);
        return response()->json([
            'cost_type' => $costType,
        ]);
    }
}
